/*
 * MapArea.h
 *
 * Created 8/13/2009 By Johnny Huynh
 *
 * Version 00.00.01 8/13/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef MAP_AREA_H
 #define MAP_AREA_H
 
 template <typename T> class MapArea;
 
 // For collision handling
 #include "CollisionHandlerEventNode.h"
 
 #include "global.h"
 #include "ObjectCollection.h"
 #include "CollisionHandlerEventNodeCollection.h"
 
 #include "collisionTraverser.h"
 #include "geoMipTerrain.h"
 #include "nodePath.h"
 #include "filename.h"
 #include "CollisionObject.h"
 
 /**
  * Class specification for MapArea
  */
 template <typename T>
 class MapArea : public CollisionObject<T>
 {
 // Data Members
 protected:
    ObjectCollection<T> _objects;
    CollisionTraverser _collision_traverser; // Colliders contained inside of _collision_traverser become "from" objects
    GeoMipTerrain _terrain;
 
 // Local Functions
 public:
    MapArea( NodePath& parent, const NodePath& focal_point, const std::string& map_area_name, const std::string& height_field_file_name );
    MapArea( const MapArea<T>& map_area );
    virtual ~MapArea();
    inline MapArea<T>& operator=( const MapArea<T>& map_area );
    virtual inline void add_object( Object<T> * object_Ptr );
    virtual inline void clear_objects();
    virtual inline CollisionTraverser& get_collision_traverser();
    virtual inline Object<T>* get_object( const std::string& key );
    virtual inline Object<T>* get_object( const NodePath& node_path );
    //inline NodePath node_path();
    virtual inline void populate_map_area( WindowFramework& window ) = 0;
    virtual inline void remove_object( const std::string& key );
    virtual inline void remove_object( Object<T> * object_Ptr );
    virtual inline void remove_object( NodePath& object_node_path );
    virtual inline void reparent_to( NodePath& node_path );
    
    // Object<T> functions
    virtual inline void handle_from_collision( Object<T>& into, const CollisionEntry& c_entry ) {}
    virtual inline void handle_into_collision( Object<T>& from, const CollisionEntry& c_entry ) {}
    virtual inline void interrupt_action() {}
    virtual inline bool is_effected_by_gravity() const { return false; }
    virtual inline void load_model( WindowFramework& window, const std::string& filename ) {}
    virtual inline void load_model( WindowFramework& window, const std::string& filename, const NodePath& parent ) {}
    virtual inline void look_at_xy_direction( const T& x, const T& y ) {}
    virtual inline void stun() {}
    virtual inline void unstun() {}
    virtual inline void translate_in_direction_of_heading( const T& heading_degrees, const T& units ) {}
    virtual inline void translate( const T& x, const T& y ) {}
    virtual inline void translate( const VECTOR2_TYPE& ground_direction ) {}
    virtual inline void translate( const T& x, const T& y, const T& z ) {}
    virtual inline void translate( const VECTOR3_TYPE& direction ) {}
 
 // Protected Functions
 protected:
    virtual inline void set_boundaries( const T width, const T height );
    
 // Private Functions
 private:
    virtual inline void set_level_of_detail();
    
 // Friend Functions
 public:
    template <typename T> friend inline AsyncTask::DoneStatus update_terrain( GenericAsyncTask* task_Ptr, void* data_Ptr );
 };
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 MapArea<T>::MapArea( NodePath& parent, const NodePath& focal_point, const std::string& map_area_name, const std::string& height_field_file_name )
            : CollisionObject<T>(),
              _collision_traverser(),
              _terrain( map_area_name )
 {
    _terrain.set_heightfield( Filename( height_field_file_name ) );
    
    reparent_to( parent );
    set_level_of_detail(); // set the LOD settings for the GeoMipTerrain
    _terrain.set_focal_point( focal_point ); // set the focal point of the terrain generation
    _terrain.generate(); // generate the terrain
    NodePath::operator=( _terrain.get_root() );
    
    global::_task_mgr_Ptr->add( new GenericAsyncTask( map_area_name /*_terrain.get_root().get_name()*/, update_terrain<T>, static_cast<void*>( this ) ) );
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 MapArea<T>::MapArea( const MapArea<T>& map_area )
            : CollisionObject<T>( map_area ),
              _objects( map_area._objects ),
              _collision_traverser( map_area._collision_traverser ),
              _terrain( map_area._terrain )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 MapArea<T>::~MapArea()
 {
    _terrain.get_root().detach_node(); // remove the terrain NodePath from its parent
    MapArea<T>::clear_objects(); // detach the objects belonging to this MapArea so that they 
                                 // no longer point to the GeoMipTerrain
    
    global::_task_mgr_Ptr->remove( global::_task_mgr_Ptr->find_task( _terrain.get_root().get_name() ) );
    global::_task_mgr_Ptr->remove( global::_action_task_mgr_Ptr->find_task( _terrain.get_root().get_name() ) );
 }
 
 /**
  * operator=() copies the content of the specified MapArea to this MapArea.
  *
  * @param (const MapArea<T>&) map_area
  * @return MapArea<T>&
  */
 template <typename T>
 inline MapArea<T>& MapArea<T>::operator=( const MapArea<T>& map_area )
 {
    CollisionObject<T>::operator=( map_area );
    _objects = map_area._objects;
    _collision_traverser = map_area._collision_traverser;
    _terrain = map_area._terrain;
    
    return *this;
 }
 
 /**
  * add_object() adds the specified object to this MapArea.
  *
  * @param (Object<T> *) object_Ptr
  */
 template <typename T>
 inline void MapArea<T>::add_object( Object<T> * object_Ptr )
 {
    // Position the Object relative to this MapArea
    NodePath terrain_node( _terrain.get_root() );
    //object_Ptr->set_pos( object_Ptr->get_pos() + terrain_node.get_pos() );
    //object_Ptr->set_hpr( object_Ptr->get_hpr() + terrain_node.get_hpr() );
    //object_Ptr->reparent_to( terrain_node ); // Not a good idea because of level of detail (LOD) and other properties
                                               // such as color and scaling carrying over
    object_Ptr->reparent_to( terrain_node.get_parent() );
    
    _objects.add( object_Ptr );
 }
 
 /**
  * clear_objects() removes all Objects in this MapArea.
  */
 template <typename T>
 inline void MapArea<T>::clear_objects()
 {
    // Detach all Objects belonging to this MapArea
    ObjectCollection<T>::iterator end_Itr( _objects.end() );
    for ( ObjectCollection<T>::iterator obj_Itr( _objects.begin() ); obj_Itr != end_Itr; ++obj_Itr )
    {
        obj_Itr->detach_node();
    }
    
    _objects.clear();
 }
 
 /**
  * get_collision_traverser() returns a reference to the collision traverser 
  * of this MapArea.
  *
  * @return CollisionTraverser&
  */
 template <typename T>
 inline CollisionTraverser& MapArea<T>::get_collision_traverser()
 {
    return _collision_traverser;
 }
 
 /**
  * get_object() returns a pointer to the Object that has the specified key.
  *
  * @param (const std::string&) key
  * @return Object<T>*
  */
 template <typename T>
 inline Object<T>* MapArea<T>::get_object( const std::string& key )
 {
    // if the key is equal to the key of this MapArea, then return this MapArea
    if ( key == StringConversion::to_str(NodePath::get_key()) )
        return this;
    
    return _objects.get( key );
 }
 
 /**
  * get() returns a pointer to the Object matching the specified node_path.
  * If the specified node_path is not an Object contained within this MapArea,
  * then NULL is returned.
  *
  * @param (const NodePath&) node_path
  * @return Object<T>*
  */
 template <typename T>
 inline Object<T>* MapArea<T>::get_object( const NodePath& node_path )
 {
    return _objects.get( node_path );
 }
 
 /**
  * node() returns a NodePath to the GeoMipTerrain of this MapArea.
  * @return NodePath
  */
 //template <typename T>
 //inline NodePath MapArea<T>::node_path()
 //{
 //   return _terrain.get_root();
 //}
 
 /**
  * remove_object() removes the Object that has the specified key from this MapArea.
  *
  * @param (const std::string&) key
  */
 template <typename T>
 inline void MapArea<T>::remove_object( const std::string& key )
 {
    MapArea<T>::remove_object( MapArea<T>::get_object( key ) );
    //_objects.remove( key );
 }
 
 /**
  * remove_object() removes the specified Object from this MapArea.
  *
  * @param (Object<T> *) object_Ptr
  */
 template <typename T>
 inline void MapArea<T>::remove_object( Object<T> * object_Ptr )
 {
    if ( object_Ptr != NULL )
    {
        object_Ptr->detach_node();
        _objects.remove( *object_Ptr );
    }
 }
 
 /**
  * remove_object() removes the Object associated with the specified 
  * NodePath from this MapArea.
  *
  * @param (NodePath&) object_node_path
  */
 template <typename T>
 inline void MapArea<T>::remove_object( NodePath& object_node_path )
 {
    object_node_path.detach_node();
    _objects.remove( object_node_path );
 }
 
 /**
  * reparent_to() sets this MapArea to be a child of the specified NodePath.
  *
  * @param (NodePath&) node_path
  */
 template <typename T>
 inline void MapArea<T>::reparent_to( NodePath& node_path )
 {
    _terrain.get_root().reparent_to( node_path );
 }
 
 /** PROTECTED FUNCTIONS **/
 
 /**
  * set_boundaries() sets the boundaries for this MapArea based on the specified width and height of the height-map.
  *
  * @param (const T) width
  * @param (const T) height
  */
 template <typename T>
 inline void MapArea<T>::set_boundaries( const T width, const T height )
 {
    PT(CollisionHandlerEventNode<T>) boundary_Ptr( new CollisionHandlerEventNode<T>( "MapArea Boundary", false ) ); // false to turn off from-collision
    CollisionObject<T>::add_collider( boundary_Ptr );
    
    T actual_width( width * NodePath::get_sx() );
    T actual_height( height * NodePath::get_sy() );
    VECTOR3_TYPE pos( NodePath::get_pos() );
    VECTOR3_TYPE b( pos + VECTOR3_TYPE(ZERO, ZERO, ONE) ); // up-vector
    VECTOR3_TYPE c( pos + VECTOR3_TYPE(ZERO, -ONE, ZERO) );
    VECTOR3_TYPE d( pos + VECTOR3_TYPE(ONE, ZERO, ZERO) );
    VECTOR3_TYPE e( pos + VECTOR3_TYPE(actual_width, actual_height, ZERO) );
    VECTOR3_TYPE f( b + e - pos );
    VECTOR3_TYPE g( e - c + pos );
    VECTOR3_TYPE h( e - d + pos);
    boundary_Ptr->collision_node()->add_solid( new CollisionPlane( Planef( pos, b, c ) ) );
    boundary_Ptr->collision_node()->add_solid( new CollisionPlane( Planef( pos, b, d ) ) );
    boundary_Ptr->collision_node()->add_solid( new CollisionPlane( Planef( e, f, g ) ) );
    boundary_Ptr->collision_node()->add_solid( new CollisionPlane( Planef( e, f, h ) ) );
    boundary_Ptr->show();
    boundary_Ptr->reparent_to( NodePath::get_parent() );
 }
 
 /** PRIVATE FUNCTIONS **/
 
 /**
  * set_level_of_detail() applies the level of detail (LOD) settings to the GeoMipTerrain.
  */
 template <typename T>
 inline void MapArea<T>::set_level_of_detail()
 {
    //_terrain.set_bruteforce( true ); // Level-of-Detail (LOD) is not applied if bruteforce is set to true
    
    // if bruteforce is false (using LOD)
    _terrain.set_min_level( 2 ); // optional - the higher the min level, the lower the quality for near view/sight
    _terrain.set_block_size( 64 ); // must be a power of 2 (e.g. 32)
    //_terrain.set_near( 150.0 );//25.0 );
    //_terrain.set_far( 250.0 );//35.0 );
    _terrain.set_near_far( 150.0, 250.0 );
    
    // if too many meshes (nodes), reduce the NodePath to only one node
    //_terrain.set_auto_flatten( GeoMipTerrain::AFM_strong );
    
    //_terrain.get_root().analyze();
 }
 
 /** FRIEND FUNCTIONS **/
 
 /**
  * update_terrain() updates the specified terrain pointed to by user_data of the GenericAsyncTask.
  * The terrain is only regenerated if it needs to be.
  *
  * @param (GenericAsyncTask*) task_Ptr
  * @param (void*) data_Ptr
  * @return AsyncTask::DoneStatus
  */
 template <typename T>
 inline AsyncTask::DoneStatus update_terrain( GenericAsyncTask* task_Ptr, void* data_Ptr )
 {
    MapArea<T>* map_area_Ptr( static_cast<MapArea<T>*>( task_Ptr->get_user_data() ) );
    GeoMipTerrain& terrain( map_area_Ptr->_terrain );
    
    terrain.update();
    
    // Set the object to be above the terrain
    NodePath& terrain_node( terrain.get_root() );
    ObjectCollection<T> object_collection( map_area_Ptr->_objects );
    ObjectCollection<T>::iterator _object_Itr( object_collection.begin() );
    while ( _object_Itr != object_collection.end() )
    {
        T terrain_z( terrain.get_elevation( _object_Itr->get_x() - terrain_node.get_x(), _object_Itr->get_y() - terrain_node.get_y() ) * terrain_node.get_sz() );
        //if ( _object_Itr->get_z() < terrain_z )
        if ( _object_Itr->is_effected_by_gravity() )
            _object_Itr->set_z( terrain_z );
        
        ++_object_Itr;
    }
    
    return AsyncTask::DS_cont;
 }
 
 #endif // MAP_AREA_H